<?php
/**
 * SharkGate - SharkActiveDefence
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 *
 * @category    SharkGate
 * @package     SharkGate_SharkActiveDefence
 * @author      Nikola Jocic
 * @copyright   Copyright (c) 2017 SharkGate LTD (http://www.sharkgate.net)
 * @license     http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

class FileMonitor
{

    /**
     * Checks is PHP extension loaded on the server
     *
     * @return boolean
     */
    public function isExtensionLoaded($extName) {
        return extension_loaded($extName);
    }    

    /**
     * Get excludes from OHSF server with curl - file_get_contents" is discouraged, but leave it as backup option.
     *
     * @return array
     */
    public function getExcludes($URL) {
        if( $this->isExtensionLoaded('allow_url_fopen') ) {
            $content = $this->getContentsMethod($URL);
        } else {
            $content = $this->curlMethod($URL);
        }
            $content = preg_split('/\r\n|\r|\n/', $content);
            return $content;  
    }

    /**
     * file_get_contents Method
     *
     * @return mixed
     */
    public function getContentsMethod($URL) {
        return file_get_contents($URL);
    }    

    /**
     * Curl Method
     *
     * @return mixed
     */
    public function curlMethod($URL) {
        $ch = curl_init();   

        curl_setopt ($ch, CURLOPT_URL, $URL);
        curl_setopt ($ch, CURLOPT_CONNECTTIMEOUT, 50); 
        curl_setopt ($ch, CURLOPT_RETURNTRANSFER, true);
        
        return $content = curl_exec($ch);
    }  

    /**
     * Send archive to scanner
     *
     * @return boolean
     */
    public function sendToScanner($archive) {

        $ftp_server = "hostname";
        $ftp_conn   = ftp_connect($ftp_server) or die("Could not connect to $ftp_server");
        $login      = ftp_login($ftp_conn, "username", "password");

        if (ftp_put($ftp_conn, $archive, getcwd()."/".$archive, FTP_ASCII)) {
           return true;
        }
        return false;

        ftp_close($ftp_conn);            
    }

    /**
     * Merge files or directories in one array and prepare to be excluded from scan
     *
     * @return array
     */
    public function zipFiles($files, $zipName) {

        $zip        =    new ZipArchive;
        $files      =    array_unique($files);
        $zip->open($zipName, ZipArchive::CREATE);

        foreach ($files as $file) {
            $zip->addFile($file);
        }
        $zip->close();

        if (file_exists(getcwd()."/".$zipName)) {
            return true;
        }
        return false;
    }

    /**
     * AutoDestruct FileMonitor from the server if we don`t need it here anymore
     *
     * @return boolean
     */
    public function autoDestruct($path) {

        $recursDirIterator = new RecursiveDirectoryIterator($path, FilesystemIterator::SKIP_DOTS);
        $recursIterator    = new RecursiveIteratorIterator($recursDirIterator, RecursiveIteratorIterator::CHILD_FIRST);

        foreach ( $recursIterator as $file ) {
            $file->isDir() ?  rmdir($file) : unlink($file);
        }
        return true;
    }

    /**
     * Merge files or directories in one array and prepare to be excluded from scan
     *
     * @return array
     */
    public function prepareExcludes($merge) {

        if ($merge == "directories") {

            $getFilesAndDirs = array(
                                "/SharkGate/inc/class.RecursiveCallbackFilterIterator.php",              
                                "/SharkGate/inc/class.FileMonitor.php",
                                "/SharkGate/inc/class.MailGun.php",
                                "/SharkGate/SharkgateMonitor.php",
                                "/SharkGate/data"
                                );
            //Pull excludes from OHSF server
            $excludeDirsRemote  =    $this->getExcludes('http://operations.onehoursitefix.com/monitor/?get=excludes');

            foreach ($excludeDirsRemote as $remoteDir) {
                $ex  = trim($remoteDir);
                $ex  = str_replace("*", "", $ex);
                if ($ex !== $this->getWebroot()) {
                    $excDirsRemote[] = $ex;
                }
            }        

            $excludeDirs    =    array_merge($getFilesAndDirs, $excDirsRemote);
            // Remote + Manual Dirs
            $excludeDirs    =    preg_replace('/^/', $this->getWebroot(), $excludeDirs);        

            foreach ($excludeDirs as $dirF) {
                if ($dirF !== $this->getWebroot()) {
                    $excDirs[] = $dirF;
                }
            }         
            return $excDirs; 
        }

        if ($merge == "extensions") { 
            $getExtensions      =    array("css","xml","sql","zip","tar.gz","ini","db","ico","md","json","txt","jpg","jpeg","pdf","xls","png");
            $excludeExtRemote   =    $this->getExcludes('http://operations.onehoursitefix.com/monitor/?get=extensions'); 
            $excludeExt         =    array_merge($getExtensions, $excludeExtRemote);     
            return $excludeExt;       
        }
        return false;
    }

    /**
     * Send email notification to store owner
     *
     * @return boolean
     */
    public function mailgunEmail($report, $subject = null) {

        $addressTo    =   "T2 Team <filemonitor@onehoursitefix.com>";

        if($subject == null){
            $subject  =   "SharkAlert: Changes Detected!";
        }
        $send = new MailgunAPI($addressTo, $subject, $report);
    }

    /**
     * Get Webroot Path
     *
     * @return string
     */
    public function getWebroot() {
        return substr((getcwd()), "0", "-10");
    }

    /**
     * Create array of files and folders and hash each file
     *
     * @return array
     */
    public function listFilesAndFolders() {

        $excludeExt   =  $this->prepareExcludes("extensions");
        $excludeDirs  =  $this->prepareExcludes("directories");

        $directories = new \RecursiveDirectoryIterator($this->getWebroot(), \RecursiveDirectoryIterator::SKIP_DOTS);
        $filter      = new \RecursiveCallbackFilterIterator(
            $directories,
            function ($current, $key, $iterator) use ($excludeExt, $excludeDirs) {
                if($iterator->hasChildren()) {
                    return true;
                }       
                foreach ($excludeDirs as $ignore) {
                    if (strpos($current->getPathName(), $ignore) !== false) {
                        return false;
                    }
                }
                return true;
            }
        );

        $iterator = new \RecursiveIteratorIterator($filter);
        foreach ($iterator as $pathname => $object) { 
            $extension  =  pathinfo($iterator->key(), PATHINFO_EXTENSION); 
            if ( !in_array($extension, $excludeExt) && !in_array($pathname, $excludeDirs) ) {
                $files[$iterator->key()] = hash_file("sha1", $iterator->key());
            }
        }
        return $files; 
    }

    /**
     * Define website`s URL
     *
     * @return string
     */
    public function getUrl($domainNameOnly =  null) {  
        $url = sprintf("%s://%s%s", isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] != 'off' ? 'https' : 'http', $_SERVER['SERVER_NAME'], $_SERVER['REQUEST_URI']);               
        $url = str_replace("/SharkGate/SharkgateMonitor.php?scan=sharkgate-scan", "", $url);    

        if ($domainNameOnly == "short"){
            return preg_replace("(^https?://)", "", $url );
        }
        return $url;                  
    }

    /**
     * Scan files and folders, compare changes and send email
     *
     * @return boolean
     */
    public function scan() {

        $updateFile    =    true;
        $getSiteURL   =    $this->getUrl();
        $getData       =    "./data";
        // Get List of Files
        $files         =    $this->listFilesAndFolders();

        // Get Data File Content
        $result        =    unserialize(file_get_contents($getData));

        // If data is empty, this should be fresh install of FM >> Put content in data and email the site owner
        if (empty($result)) {
            foreach ($files as $path => $hash) {
                $data[] = array(
                    'file_path' => $path,
                    'file_hash' => $hash
                );
            }

            $result = file_put_contents($getData, serialize($data));
            $report = "<h4>URL: ".$getSiteURL."</h4>
                       <h3>FIrst scan completed</h3>
                       <p> - <i>Sote files scanned</i></p>
                       <p> - <i>List of files created</i></p>
                       <h4>Next time changes are made, T2 will be notified.</h4>
                       <h4 style='color:red;'>New version test - hopefully <b>no more false positives. :)</b></h4>";

            $this->mailgunEmail($report, "SharkAlert: File Monitor Installed!"); 

        } else {
            
            foreach ($result as $value) {
                $scanned[$value["file_path"]] = $value["file_hash"];
            }

            $diffs = array_diff_assoc($files, $scanned);
            unset($scanned);

            //  sort differences into Deleted, Altered and Added arrays
            if (!empty($files)) {
                if (!empty($result)) {
                    // Scanned will be old scan
                    $scanned = array();

                    // Difference between $files and $scanned
                    $diffs = array();

                    // foreach $files to get values
                    foreach ($result as $value) {
                        if (!array_key_exists($value["file_path"], $files)) {
                            //  Check for Deleted files
                            $diffs["Files Removed"][$value["file_path"]] = $value["file_path"];
                            $scanned[$value["file_path"]] = $value["file_hash"];
                        } else {
                            //  Check for Changed files
                            if ($files[$value["file_path"]] <> $value["file_hash"]) {
                                $diffs["Files Changed"][$value["file_path"]] = $value["file_path"];
                                $scanned[$value["file_path"]] = $value["file_path"];
                            } else {
                                //  Everything else should be Unchanged
                                $scanned[$value["file_path"]] = $value["file_hash"];
                            }
                        }
                    }

                    if (count($scanned) < count($files)) {
                        // Check for Added files
                        $diffs["Files Added"] = array_diff_assoc($files, $scanned); 
                    }
                    unset($scanned);
                }
            }

            //  display discrepancies
            if (!empty($diffs)) {
                $report = "<h4 style='color:red;'>New version test - hopefully <b>no more false positives. :)</b></h4>
                           <h4>Store URL: ".$getSiteURL."</h4><p></p><b>Changes detected:</b>";
                foreach ($diffs as $status => $affected) { 
                    if (is_array($affected) && !empty($affected)) {
                        $report .= "<br><br><li>" .$status. "</li>";
                        foreach ($affected as $path => $hash) {
                            // Cut the string and make it easier for T2 to copy and paste paths for excludes
                            $path = str_replace($this->getWebroot(), "", $path);
                            $report .= "<br>" .$path;

                            // If there are modified or added files, prepare them to be zipped and sent to scanner
                            if($status == "Files Changed" || $status == "Files Added"){
                                $toBeZipped[] = $this->getWebroot().$path;
                            }                            
                        }
                    }
                } 

                        /* Commented for this version
                        if (!empty($toBeZipped)) {
                            $zipName = "FileMonitor_".$this->getUrl('short')."_".str_replace("-", "_", date("Y-m-d")).".zip";
                            if($this->zipFiles($toBeZipped, $zipName)){
                                $this->sendToScanner($zipName);
                            } else {
                                $this->mailgunEmail("Error: Failed to create archive.", "ERROR");
                            }
                        }
                        */

            } else {
                $updateFile = false;
                $report = false;
            }
            // If $updateFile is set to true update data file
            if ($updateFile) {
                foreach ($files as $path => $hash) {
                        $data[] = array(
                        'file_path' => $path,
                        'file_hash' => $hash
                    );
                }
                // Save Data for the next scan
                file_put_contents($getData, serialize($data));
            }
        }        
        // If there are changes detected, email t2 members
        if ($report) {
            echo $report;
            $this->mailgunEmail($report); 
        }
        return false;
    }

}